/******************************************************************************
 ** $Id: Kalkulator.java 3634 2024-12-24 16:53:00Z wmh $
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package ebflmaennle.berechnung;

import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;

import javax.swing.Action;
import javax.swing.SwingUtilities;
import javax.swing.SwingWorker;
import javax.swing.Timer;

import ebflmaennle.Rundschreiben;
import ebflmaennle.Verfolgung;
import ebflmaennle.oberflaeche.Delegation;
import ebflmaennle.oberflaeche.Leinwand;
import ebflmaennle.oberflaeche.Leinwand.Zeiger;

public final class Kalkulator extends PropertyChangeSupport
        implements ActionListener, PropertyChangeListener, Delegation.Beantragung {

    public final record Parameter(Charakteristik charakteristik, Point2D ort) {
    }

    private final class Schnappschuss extends SwingWorker<Long, Void> {

        private final Antrag antrag;
        private final Rectangle2D sichtfenster;
        private final long startzeit;
        private int iterationsmaximum = 0;

        Schnappschuss(final Antrag antrag) {
            this.antrag = antrag;
            sichtfenster = new Rectangle2D.Double(antrag.konfiguration.a0, antrag.konfiguration.b0,
                    antrag.rechengroesse.width * antrag.konfiguration.pxs,
                    antrag.rechengroesse.height * antrag.konfiguration.pxs);
            startzeit = System.nanoTime();
        }

        @Override
        public Long doInBackground() {
            final var m = antrag.rechengroesse.width;
            final var n = antrag.rechengroesse.height;
            karte.leere(m, n);
            if (altantrag == null || m != altantrag.rechengroesse.width || n != altantrag.rechengroesse.height) {
                algorithmus.verankere(m, n, antrag.konfiguration.antialiasing, antrag.konfiguration.vorschau);
            }
            iterationsmaximum = algorithmus.bearbeite(antrag.konfiguration);
            return dauer();
        }

        @Override
        public void done() {
            stopper.stop();
            schnappschuss = null;
            var rechendauer = (long) 0;
            try {
                rechendauer = get();
            } catch (CancellationException | InterruptedException | ExecutionException exception) {
                exception.printStackTrace();
            }
            meldeFertigstellung(antrag.konfiguration, iterationsmaximum, rechendauer);
            final var ausnahme = rechendauer < Stopper.ANTRAGSVERZOEGERUNG_NS;
            Kalkulator.this.altantrag = antrag;
            Kalkulator.this.iterationsmaximum = iterationsmaximum;
            Kalkulator.this.rechendauer = rechendauer;
            if (antrag != Kalkulator.this.zielantrag || wiedervorlage || abschluss) {
                if (abschluss) {
                    beschleunigt = false;
                    abschluss = false;
                } else {
                    beschleunigt = beschleunigt || ausnahme;
                    if (wiedervorlage) {
                        abschluss = beschleunigt;
                        wiedervorlage = false;
                    } else if (!abschluss) {
                        wiedervorlage = beschleunigt;
                        if (wiedervorlage) {
                            stopper.vormerke();
                        }
                    }
                }
                final var rechengroesse = ausschnitt.rechengroesse();
                final var konfiguration = ausschnitt.konfiguration(option, beschleunigt);
                Kalkulator.this.zielantrag = new Antrag(rechengroesse, konfiguration);
                beantrage(Kalkulator.this.zielantrag);
            } else {
                ausschnitt.adaptiere(option.qualitaet, iterationsmaximum);
            }
        }

        /** @return Dauer in Millisekunden seit Start der Berechnung. */
        long dauer() {
            return System.nanoTime() - startzeit;
        }
    }

    private final class Stopper extends Timer {
        private static final long serialVersionUID = 1300426351018226247L;

        private static final int ANTRAGSVERZOEGERUNG_MS = 250;
        private static final int MS_IN_NS = 1_000_000;
        private static final long ANTRAGSVERZOEGERUNG_NS = ANTRAGSVERZOEGERUNG_MS * MS_IN_NS;

        /** Gibt an, ob der aktuelle Schnappschuss am Rechnen ist. */
        private boolean rechnend;
        /** Gibt an, ob ein Stoppauftrag abgesetzt wurde, bevor die Berechnung beendet wurde. */
        private boolean beauftragt;

        Stopper(final ActionListener listener) {
            super(0, listener);
            setRepeats(false);
        }

        @Override
        public void stop() {
            super.stop();
            beauftragt = false;
        }

        private void beauftrage(final long dauer) {
            if (!beauftragt && !isRunning()) {
                setInitialDelay(Integer.max(0, (int) (ANTRAGSVERZOEGERUNG_NS - dauer) / MS_IN_NS));
                restart();
            }
        }

        void bestaetige() {
            beschleunigt = !rechnend;
            beauftragt = true;
        }

        void vormerke() {
            rechnend = true;
            beauftrage(0);
        }

        void vormerke(final long dauer) {
            rechnend = false;
            beauftrage(dauer);
        }
    }

    private final class Ermittlung extends SwingWorker<Parameter, Void> {
        private final Parameter parameter;
        private Parameter zentrum;
        private Zeiger zeiger;
        private volatile boolean verarbeitet;

        Ermittlung(final Parameter parameter, final Ermittlung vorgaenger) {
            this.parameter = parameter;
            if (vorgaenger != null) {
                vorgaenger.verarbeitet = true;
                zeiger = vorgaenger.zeiger;
                zentrum = vorgaenger.zentrum;
            }
        }

        @Override
        protected Parameter doInBackground() throws Exception {
            return zentrum(parameter);
        }

        @Override
        protected void done() {
            try {
                final var zentrum = get();
                if (verarbeitet) {
                    return;
                }
                propagiere(zentrum);
            } catch (InterruptedException | ExecutionException ausnahme) {
                ausnahme.printStackTrace();
            }
        }

        void propagiere(final Parameter zentrum) {
            melde(zentrum);
            Zeiger zeiger = null;
            if (zentrum != null) {
                final var nachfolgerort = ausschnitt.darstellungspunkt(zentrum.ort());
                zeiger = new Zeiger(nachfolgerort, null, 0);
            }
            melde(zeiger);
            this.zentrum = zentrum;
            this.zeiger = zeiger;
        }

        void melde(final Parameter zentrum) {
            verarbeitet = true;
            if (this.zentrum != null || zentrum != null) {
                Kalkulator.this.firePropertyChange("Parameter", this.zentrum, zentrum);
            }
        }

        private void melde(final Zeiger zeiger) {
            if (this.zeiger != null || zeiger != null) {
                Kalkulator.this.firePropertyChange("Zentrum", this.zeiger, zeiger);
            }
        }

        private Parameter zentrum(final Parameter parameter) {
            final var n = algorithmus.nukleus(parameter);
            if (n == null) {
                return null;
            }
            final var charakteristik = new Charakteristik(this.parameter.charakteristik().nu, n.lq, 0);
            return new Parameter(charakteristik, n.ort());
        }
    }

    private record Antrag(Dimension rechengroesse, Konfiguration konfiguration) {
    }

    private static final long serialVersionUID = -5583626198199185012L;

    private final Rundschreiben<Kalkulator> rundschreiben = new Rundschreiben<>(this);
    private final Verfolgung<Kalkulator> verfolgung = new Verfolgung<>(this);
    private final Stopper stopper = new Stopper(this);
    private final Option option;
    private final Karte karte;
    private final Charakteristik[] charakteristiken;
    private final Algorithmus algorithmus;
    private final Ausschnitt ausschnitt;

    /** Entscheidet darüber, ob der Algorithmus anstehende Zielkonfigurationen beschleunigt berechnen soll. */
    private boolean beschleunigt;
    /** Gibt an, ob eine Wiedervorlage der aktuellen Schnappschusskonfiguration nötig ist. */
    private boolean wiedervorlage;
    /** Gibt an, ob eine Konfiguration abschließend nochmals berechnet werden muss. */
    private boolean abschluss;
    private Schnappschuss schnappschuss;
    private Ermittlung ermittlung;
    private Rectangle2D sichtfenster;
    private Antrag altantrag;
    private Antrag zielantrag;
    private long rechendauer;
    private int iterationsmaximum = -1;

    public Kalkulator(final Option option, final Karte karte, final Ausschnitt ausschnitt, final Leinwand leinwand) {
        super(option);
        this.option = option;
        this.karte = karte;
        this.charakteristiken = karte.charakteristiken;
        this.ausschnitt = ausschnitt;
        algorithmus = new Algorithmus(charakteristiken, leinwand);
    }

    @Override
    public void actionPerformed(final ActionEvent ereignis) {
        assert schnappschuss != null;
        ((Stopper) ereignis.getSource()).bestaetige();
        algorithmus.abbreche();
        firePropertyChange("Stopp", false, true);
    }

    @Override
    public void propertyChange(final PropertyChangeEvent ereignis) {
        assert SwingUtilities.isEventDispatchThread();
        if ("Parameter".equals(ereignis.getPropertyName())) {
            final var parameter = (Parameter) ereignis.getNewValue();
            if (parameter != null) {
                ermittlung = new Ermittlung(parameter, ermittlung);
                ermittlung.execute();
            } else if (ermittlung != null) {
                ermittlung.propagiere(null);
            }
        } else if ("Import".equals(ereignis.getPropertyName())) {
            beantrage();
        }
    }

    public void registriere(final Action aktivierung, final PropertyChangeSupport aenderungsverursacher) {
        verfolgung.registriere(aktivierung, aenderungsverursacher);
    }

    public Algorithmus algorithmus() {
        return algorithmus;
    }

    @Override
    public void beantrage() {
        // Bei einem externen Neuauftrag muss die Zielrechengröße beachtet werden.
        final var rechengroesse = ausschnitt.rechengroesse();
        // Bei einem externen Neuauftrag muss die Zielkonfiguration neu erstellt werden.
        final var konfiguration = ausschnitt.konfiguration(option, beschleunigt);
        // Erstelle den neuen Antrag.
        zielantrag = new Antrag(rechengroesse, konfiguration);
        // Ein externer Neuauftrag macht eine Wiedervorlage obsolet, überdies entscheidet der Schnappschuss.
        wiedervorlage = false;
        // Für den Abschluss gilt dieselbe Überlegung wie für die Wiedervorlage.
        abschluss = false;
        if (schnappschuss != null) {
            stopper.vormerke(schnappschuss.dauer());
        } else {
            beantrage(zielantrag);
        }
    }

    public void abbreche() {
        stopper.vormerke();
    }

    public Rundschreiben<Kalkulator> rundschreiben() {
        return rundschreiben;
    }

    public Parameter parameter(final Zeiger zeiger) {
        if (zeiger == null || zeiger.position() == null) {
            return null;
        }
        final var p = ausschnitt.p(zeiger.position());
        return new Parameter(charakteristiken[p], ausschnitt.ort(p));
    }

    private void beantrage(final Antrag antrag) {
        schnappschuss = new Schnappschuss(antrag);
        schnappschuss.execute();
        meldeSchnappschuss();
    }

    private void meldeSchnappschuss() {
        firePropertyChange("Antialiasing",
                altantrag == null ? null : Integer.valueOf(altantrag.konfiguration.antialiasing),
                Integer.valueOf(schnappschuss.antrag.konfiguration.antialiasing));
        firePropertyChange("Vorschau", altantrag == null ? null : Integer.valueOf(altantrag.konfiguration.vorschau),
                Integer.valueOf(schnappschuss.antrag.konfiguration.vorschau));
        firePropertyChange("Bildgröße", altantrag == null ? null : altantrag.rechengroesse,
                schnappschuss.antrag.rechengroesse);
        firePropertyChange("Sichtfenster", sichtfenster == null ? null : sichtfenster, schnappschuss.sichtfenster);
        firePropertyChange("Auflösung", altantrag == null ? null : Float.valueOf(altantrag.konfiguration.pxs),
                Float.valueOf(schnappschuss.antrag.konfiguration.pxs));
        firePropertyChange("Bevorzugung",
                altantrag == null ? Option.BEVORZUGUNGSSTANDARD : altantrag.konfiguration.bevorzugung,
                schnappschuss.antrag.konfiguration.bevorzugung);
        firePropertyChange("Qualitaet",
                altantrag == null ? Option.QUALITAETSSTANDARD : altantrag.konfiguration.qualitaet,
                schnappschuss.antrag.konfiguration.qualitaet);
        firePropertyChange("Iterationslimit", altantrag == null ? 0 : altantrag.konfiguration.iterationslimitoptimum,
                schnappschuss.antrag.konfiguration.iterationslimitoptimum);
        firePropertyChange("Berechnung", false, true);

    }

    private void meldeFertigstellung(final Konfiguration konfiguration, final int iterationsmaximum, final long dauer) {
        firePropertyChange("Iterationsmaximum", this.iterationsmaximum, iterationsmaximum);
        firePropertyChange("Berechnung", true, false);
        firePropertyChange("Rechendauer", this.rechendauer, dauer);
        firePropertyChange("Konfiguration", altantrag == null ? null : altantrag.konfiguration, konfiguration);
        rundschreiben.fireStateChanged();
    }

}
