/******************************************************************************
 ** $Id: Dz.java 3594 2024-09-07 16:54:16Z wmh $
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package ebflmaennle.berechnung;

/**
 * Berechnet die Folge (dz[m])_m der Ableitungen dz[m] = dF°(m)/dz zur Folge (z[m])_m für den Parameter c. Die
 * Startwerte sind dz[0] = 1 und dz[1] = c.
 */
final class Dz {
    private final Z z;

    double x; /* Re dz[m]/dz */
    double y; /* Im dz[m]/dz */
    double xq; /* x^2 */
    double yq; /* y^2 */
    double absq; /* x^2+y^2 */

    Dz(final Z t) {
        z = t;
    }

    private void quadriere() {
        xq = x * x;
        yq = y * y;
        absq = xq + yq;
    }

    /** Berechnet nächstes Folgenglied dz[m+1] = 2 * z[m] * dz[m]. */
    void iteriere() {
        iteriere(z.x + z.y);
    }

    /**
     * Berechnet nächstes Folgenglied dz[m+1] = 2z[m] * dz[m] mittels Gauß-Multiplikation in Rekursion.
     *
     * @param zspur
     *            Summe des Realteils und des Imaginärteils einer komplexen Zahl
     */
    void iteriere(final double zspur) {
        final var k1 = x * zspur;
        final var k2 = z.y * (x + y);
        final var k3 = z.x * (y - x);
        x = 2 * (k1 - k2);
        y = 2 * (k1 + k3);
        quadriere();
    }

    // dz = 0
    void beginne() {
        x = 0;
        y = 0;
    }

    /** Startet Folge (dz[m]) mit dem Startwert dz[0] = 1. */
    void starte0() {
        x = 1;
        y = 0;
        quadriere();
    }

    /** . */
    void starte1(final double x, final double y) {
        this.x = 2 * x;
        this.y = 2 * y;
        quadriere();
    }

    /** Startet Folge (dz[m]) mit dem Startwert dz[1] = 2 * z[0]. */
    void starte1() {
        x = 2 * z.x;
        y = 2 * z.y;
        quadriere();
    }
}