/******************************************************************************
 ** $Id: Distanzschaetzer.java 3562 2024-08-31 16:46:51Z wmh $
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package ebflmaennle.berechnung;

abstract class Distanzschaetzer {

    private interface Lq {
        float wert(final float korrektur);
    }

    private interface Mulq {
        float wert(final float lq);
    }

    private interface Nu {
        float wert(final float korrektur, final int index);
    }

    private static final Lq lqf(final boolean schrittschaetzung) {
        // Distanz zum Rand
        final Lq lqfk = korrektur -> 1;
        // Distanz zum Band
        final Lq lqfs = korrektur -> {
            final var chi = (float) (0.5 - Math.abs(korrektur));
            final var c = ((float) Math.sqrt(2) / 16);
            return chi * chi * c;
        };
        return schrittschaetzung ? lqfs : lqfk;
    }

    private static final Mulq mulqf(final boolean schrittschaetzung, final double pxs) {
        final var pxsq = (float) (pxs * pxs / 2);
        final Mulq mulqfk = lq -> Charakteristik.MULQ_RANDNAH;
        final Mulq mulqfs = lq -> {
            if (lq > pxsq) {
                return Charakteristik.MULQ_GEFLUECHTET;
            }
            return Charakteristik.MULQ_RANDNAH;
        };
        return schrittschaetzung ? mulqfs : mulqfk;
    }

    /** Behandelt, falls ein Orbit flüchtet, also der Parameter nicht zur Mandelbrotmenge gehört. */
    private static class Platzhalterschaetzer extends Distanzschaetzer {

        private final Nu nuf;
        protected final Z z;

        Platzhalterschaetzer(final Z z, final boolean aussenblockschaetzung) {
            this.z = z;
            if (aussenblockschaetzung) {
                nuf = (korrektur, index) -> index;
            } else {
                nuf = (korrektur, index) -> index - korrektur;
            }
        }

        private float nu(final float korrektur, final int index) {
            return nuf.wert(korrektur, index);
        }

        /** Schätzt quadrierten Abstand. */
        protected float lq() {
            return Charakteristik.LQ_AUSSEN;
        }

        protected float lqBand(final float lq, final float korrektur) {
            return Charakteristik.LQ_AUSSEN;
        }

        protected float mulq(final float lq) {
            return Charakteristik.MULQ_RANDNAH;
        }

        protected boolean gefluechtet(final Charakteristik charakteristik, final int index) {
            final var gefluechet = z.gefluechtet();
            if (gefluechet) {
                final var lq = lq();
                final var korrektur = korrektur();
                charakteristik.setze(nu(korrektur, index), lqBand(lq, korrektur), mulq(lq));
                assert charakteristik.aussen();
            }
            return gefluechet;
        }

        protected float korrektur() {
            return (float) korrektur(z.logabsq());
        }

        @Override
        void initialisiere() {
            // Keine Berechnung, da hier keine Distanz geschätzt wird.
        }

        @Override
        void aktualisiere() {
            // Keine Berechnung, da hier keine Distanz geschätzt wird.
        }

        @Override
        boolean bestimmt(final Charakteristik charakteristik, final int index) {
            return gefluechtet(charakteristik, index);
        }
    }

    /* Zentrumschätzer nach Henriksen. */
    private static class Zentrumschaetzer extends Platzhalterschaetzer {
        private final Lq lqf;
        private final Mulq mulqf;
        private final double r;
        private double u;
        private double u2;
        private double v;
        private double v2;

        Zentrumschaetzer(final Z z, final double pxs, final boolean aussenblockschaetzung,
                final boolean schrittschaetzung, final boolean beschleunigt) {
            super(z, aussenblockschaetzung);
            lqf = lqf(false); // Distanzschätzung steht nicht zur Verfügung.
            mulqf = mulqf(schrittschaetzung, pxs);
            if (beschleunigt) {
                r = pxs;
            } else {
                r = pxs / Math.sqrt(2);
            }
        }

        private boolean zentral(final Charakteristik charakteristik) {
            final var gefangen = z.absq * (1 + z.absq) < u * u + v * v;
            if (gefangen) {
                charakteristik.setze(Charakteristik.NU_ZENTRUM, Charakteristik.LQ_ZENTRUM, Charakteristik.MULQ_ZENTRUM);
            }
            return gefangen;
        }

        @Override
        protected float lqBand(final float lq, final float korrektur) {
            return lq * lqf.wert(korrektur);
        }

        @Override
        protected float mulq(final float lq) {
            return mulqf.wert(lq);
        }

        /**
         * Initialisiere Distanzfolge (dc[n]) mit dem Startwert dc[2].
         * @formatter:off
         * z[0] = 0, z[1] = c; dc[0] = 0, dc[1] = r, dc[2] = 2*z[1]*dc[1] + r = 2c + r
         * @formatter:on
         */
        @Override
        void initialisiere() {
            u = (2 * z.x + 1) * r;
            v = 2 * z.y * r;
            u2 = 2 * u;
            v2 = 2 * v;
        }

        /** Iteriere mit der Rekursionsvorschrift dc[n+1] = 2*z[n]*dc[n] + r, dc[0] = 0. */
        @Override
        void aktualisiere() {
            u = z.x * u2 - z.y * v2 + r;
            v = z.x * v2 + z.y * u2;
            u2 = 2 * u;
            v2 = 2 * v;
        }

        @Override
        boolean bestimmt(final Charakteristik charakteristik, final int index) {
            return gefluechtet(charakteristik, index) || zentral(charakteristik);
        }
    }

    /**
     * Schätzt s^2 mit dem besten Schätzer den wahren quadrierten Abstand d^2 mit s/4 <= d <= s.
     * @formatter:off
     * s^2 = (2 * |z|/|dc| * log(|z|))^2 = 4 * |z|^2/|dc|^2 * log(|z|^2)^2 / 4 = |z|^2/|dc|^2 * (log(|z|^2))^2
     * @formatter:on
     */
    private static class Bestschaetzer extends Platzhalterschaetzer {
        private final Lq lqf;
        private final Mulq mulqf;
        private double logabsq;
        protected final Dc dc;

        Bestschaetzer(final Z z, final double pxs, final boolean aussenblockschaetzung,
                final boolean schrittschaetzung) {
            super(z, aussenblockschaetzung);
            lqf = lqf(schrittschaetzung);
            mulqf = mulqf(schrittschaetzung, pxs);
            dc = new Dc(z);
        }

        @Override
        protected float lq() {
            logabsq = z.logabsq();
            return (float) (dc.alpha() * logabsq * logabsq);
        }

        @Override
        protected float lqBand(final float lq, final float korrektur) {
            return lq * lqf.wert(korrektur);
        }

        @Override
        protected float mulq(final float lq) {
            return mulqf.wert(lq);
        }

        @Override
        protected float korrektur() {
            return (float) korrektur(logabsq);
        }

        @Override
        void initialisiere() {
            dc.starte2();
        }

        @Override
        void aktualisiere() {
            dc.iteriere();
        }
    }

    /** Schätzung der Distanz sowie Detektion des Randes nach Henriksen. */
    private static class Kombischaetzer extends Bestschaetzer {
        private final double rq;

        Kombischaetzer(final Z z, final double pxs, final boolean aussenblockschaetzung,
                final boolean schrittschaetzung, final boolean beschleunigt) {
            super(z, pxs, aussenblockschaetzung, schrittschaetzung);
            if (beschleunigt) {
                rq = pxs * pxs;
            } else {
                rq = pxs * pxs / 2;
            }
        }

        private boolean zentral(final Charakteristik charakteristik) {
            final var gefangen = z.absq * (1 + z.absq) < rq * dc.absq();
            if (gefangen) {
                charakteristik.setze(Charakteristik.NU_ZENTRUM, Charakteristik.LQ_ZENTRUM, Charakteristik.MULQ_ZENTRUM);
            }
            return gefangen;
        }

        @Override
        boolean bestimmt(final Charakteristik charakteristik, final int index) {
            return gefluechtet(charakteristik, index) || zentral(charakteristik);
        }
    }

    private static double korrektur(final double logabsq) {
        return Math.log(logabsq) / Math.log(2) - Charakterisierung.NULLPUNKT;
    }

    static Distanzschaetzer erzeuge(final Z z, final double pxs, final boolean aussenblockschaetzung,
            final boolean zentrumdetektion, final boolean distanzschaetzung, final boolean schrittschaetzung,
            final boolean beschleunigt) {
        final Distanzschaetzer distanzschaetzer;
        if (distanzschaetzung) {
            if (zentrumdetektion) {
                distanzschaetzer = new Kombischaetzer(z, pxs, aussenblockschaetzung, schrittschaetzung, beschleunigt);
            } else {
                distanzschaetzer = new Bestschaetzer(z, pxs, aussenblockschaetzung, schrittschaetzung);
            }
        } else if (zentrumdetektion) {
            distanzschaetzer = new Zentrumschaetzer(z, pxs, aussenblockschaetzung, schrittschaetzung, beschleunigt);
        } else {
            distanzschaetzer = new Platzhalterschaetzer(z, aussenblockschaetzung);
        }
        return distanzschaetzer;
    }

    abstract void initialisiere();

    abstract void aktualisiere();

    /**
     * Gibt an, ob der Orbit für den Parameter zum Kartenindex geflüchtet oder ein geschätztes Zentrum ist.
     *
     * @param p
     *            Kartenindex
     * @param index
     *            Index des Folgengliedes
     * @return Wahr, falls Parameter außerhalb der Mandelbrotmenge oder aber ein Zentrum ist; ansonsten falsch.
     */
    abstract boolean bestimmt(final Charakteristik charakteristik, final int index);
}