/******************************************************************************
 ** $Id $
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package ebflmaennle.berechnung;

/**
 * Berechnet die Folge (dc[m])_m der Ableitungen dc[m] = dc[m]/dc zur Folge (z[m])_m für den Parameter c. Die Startwerte
 * sind dc[0] = 1 und dc[1] = 1.
 */
final class Dc {
    private final Z z;
    double x; /* Re dz[m]/dc */
    double y; /* Im dz[m]/dc */
    double x2; /* 2x */
    double y2; /* 2y */

    Dc(final Z z) {
        this.z = z;
    }

    double absq() {
        return x * x + y * y;
    }

    double alpha() {
        return z.absq / (x * x + y * y);
    }

    /** Iteriere mit der Rekursionsvorschrift dc[n+1] = 2z[n]dc[n] + 1, dc[0] = 0. */
    void iteriere() {
        x = z.x * x2 - z.y * y2 + 1;
        y = z.x * y2 + z.y * x2;
        x2 = 2 * x;
        y2 = 2 * y;
    }

    /**
     * Initialisiere Folge (dc[n]) mit dem Startwert dc[1].
     * @formatter:off
     * z[0] = 0, z[1] = c; dc[0] = 0, dc[1] = 1
     * @formatter:on
     */
    void starte1() {
        x = 1;
        y = 0;
        x2 = 2;
        y2 = 0;
    }

    /**
     * Initialisiere Folge (dc[n]) mit dem Startwert dc[2].
     * @formatter:off
     * z[0] = 0, z[1] = c; dc[0] = 0, dc[1] = 1, dc[2] = 2 * z[1] * dc[1] + 1 = 2c + 1
     * @formatter:on
     */
    void starte2() {
        x = 2 * z.a + 1;
        y = 2 * z.b;
        x2 = 2 * x;
        y2 = 2 * y;
    }
}