/******************************************************************************
 ** $Id: Charakteristik.java 3631 2024-12-24 16:34:32Z wmh $
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package ebflmaennle.berechnung;

public final class Charakteristik {

    private static float OFFEN = Float.NaN;
    private static float UNBESTIMMT = Float.NEGATIVE_INFINITY;
    private static float GEFLUECHTET = 0;
    private static float ZENTRUM = -GEFLUECHTET;

    public static final float LQ_OFFEN = OFFEN;
    public static final float LQ_UNBESTIMMT = UNBESTIMMT;
    public static final float LQ_GEFLUECHTET = GEFLUECHTET;
    public static final float LQ_ZENTRUM = ZENTRUM;
    public static final float LQ_AUSSEN = Float.POSITIVE_INFINITY;
    public static final float MULQ_OFFEN = OFFEN;
    public static final float MULQ_GEFLUECHTET = Float.POSITIVE_INFINITY;
    public static final float MULQ_UNBESTIMMT = UNBESTIMMT;
    public static final float MULQ_ZENTRUM = ZENTRUM;
    public static final float MULQ_RANDNAH = 1;
    public static final float NU_OFFEN = OFFEN;
    public static final float NU_HAUPTKOERPER = -1;
    public static final float NU_KOPF = -2;
    public static final float NU_UNBESTIMMT = UNBESTIMMT;
    public static final float NU_ZENTRUM = ZENTRUM;

    final record Kern(float nu, float lq) {
    }

    /**
     * Kontinuierliche Fluchtzeiten. Positiv: Fluchtzeit. Negativ: Domänenperiode oder Zyklusperiode. Null: Unbestimmt.
     */
    public float nu;
    /**
     * Quadrierte Schätzwerte für Längen; positive Werte für die Außendistanz, negative Werte für den Innenabstand. Der
     * Schätzwert dist² nähert die quadrierte wahre Außendistanzen d² zur Mandelbrotmenge mit dist/4 <= d <= dist.
     */
    public float lq;
    /** Multiplikator. */
    public float mulq;

    public static boolean aussen(final float wert) {
        return wert > 0;
    }

    public static boolean innen(final float wert) {
        return wert < 0;
    }

    public static boolean offen(final float wert) {
        return Float.isNaN(wert);
    }

    public static boolean unbestimmt(final float wert) {
        return wert == UNBESTIMMT;
    }

    public static boolean zentral(final float wert) {
        return 1 / wert == Float.NEGATIVE_INFINITY;
    }

    static float koerperkopf(final double a) {
        return a > -0.75 ? NU_HAUPTKOERPER : NU_KOPF;
    }

    Charakteristik(final float nu, final float lq, final float mulq) {
        this.nu = nu;
        this.lq = lq;
        this.mulq = mulq;
    }

    Charakteristik() {
    }

    public boolean aussen() {
        return aussen(nu);
    }

    public boolean nichtinnen() {
        return nu >= 0;
    }

    public boolean offen() {
        return offen(nu);
    }

    public boolean bestimmt() {
        return nu <= 0 && lq <= 0 || nu > 0 && lq >= 0 /* Gleichheit wegen Ungenauigkeit */;
    }

    void kopiere(final Charakteristik ziel) {
        assert ziel.offen();
        ziel.nu = nu;
        ziel.lq = LQ_UNBESTIMMT;
    }

    void setze(final float nu) {
        assert offen();
        this.nu = nu;
        this.lq = LQ_UNBESTIMMT;
    }

    void leere() {
        setze(NU_OFFEN, LQ_OFFEN, MULQ_OFFEN);
    }

    void setze(final float nu, final float lq, final float mulq) {
        this.nu = nu;
        this.lq = lq;
        this.mulq = mulq;
    }

    void setze(final Kern kern, final float mulq) {
        setze(kern.nu, kern.lq, mulq);
    }

    void koerperkopf(final Z z) {
        if (nu == NU_KOPF) {
            nu = koerperkopf(z.a);
        }
    }

}