/******************************************************************************
 ** $Id: Charakterisierung.java 3631 2024-12-24 16:34:32Z wmh $
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package ebflmaennle.berechnung;

final class Charakterisierung {

    private static final int FLUCHTRADIUSMINIMUM = 2;
    private static final int FLUCHTRADIUSQUADRATFAKTOR = 16;

    static final int FLUCHTRADIUSQUADRATMINIMUM = FLUCHTRADIUSMINIMUM * FLUCHTRADIUSMINIMUM;
    static final double FLUCHTRADIUSQUADRAT = FLUCHTRADIUSQUADRATFAKTOR * FLUCHTRADIUSQUADRATMINIMUM;
    static final double NULLPUNKT = Math.log(Math.log(Charakterisierung.FLUCHTRADIUSQUADRAT)) / Math.log(2) + 0.5;

    private final Charakteristik[] charakteristiken;
    private final int m;
    private final Z z;
    private final Innendetektor innendetektor;
    private final Distanzschaetzer distanzschaetzer;
    private final Abstandschaetzer abstandschaetzer;
    private final Wukleusfinder wukleusfinder;

    private final int iterationslimit;
    private final boolean koerperkopferkennung;
    private final double a0;
    private final double b0;
    private final double pxs;
    private final boolean multiplikatoren;

    Charakterisierung(final Charakteristik[] charakteristiken, final Konfiguration konfiguration, final int m,
            final Kontext kontext) {
        this.charakteristiken = charakteristiken;
        z = kontext.z;
        innendetektor = kontext.innendetektor;
        distanzschaetzer = kontext.distanzschaetzer;
        abstandschaetzer = kontext.abstandschaetzer;
        wukleusfinder = kontext.wukleusfinder;
        this.m = m;
        iterationslimit = konfiguration.iterationslimitoptimum;
        koerperkopferkennung = konfiguration.koerperkopferkennung;
        multiplikatoren = konfiguration.multiplikatoren;
        a0 = konfiguration.a0;
        b0 = konfiguration.b0;
        pxs = konfiguration.pxs;
    }

    private double a(final int i) {
        return a0 + i * pxs;
    }

    private double b(final int j) {
        return b0 + j * pxs;
    }

    private int itmax(final float nu, final int itmax) {
        return Integer.max(itzahl(nu), itmax);
    }

    private int itzahl(final float nu) {
        final int iterationsanzahl;
        if (Float.isFinite(nu)) {
            if (nu > 0) {
                iterationsanzahl = (int) nu;
            } else {
                iterationsanzahl = -2 * (int) nu;
            }
        } else {
            iterationsanzahl = 0;
        }
        return iterationsanzahl;
    }

    // Hase-Igel-Suche nach Zyklen
    private void bestimme(final Charakteristik charakteristik) {
        var domabsq = multiplikatoren ? 1 : z.absq; // Minimaler quadrierter Abstand zum Ursprung
        var igel = 1; // Igelindex. Hier sind für diesen Iterationsindex die Iterationswerte gültig.
        var domper = igel;
        var hase = 2 * igel; // Hasenindex
        innendetektor.starte();
        distanzschaetzer.initialisiere();
        wukleusfinder.initialisiere();
        do { // Igel-Schleife
            z.merke();
            assert hase == 2 * igel;
            do { // Hase-Schleife
                innendetektor.iteriere();
                z.iteriere();
                ++igel; // Für neuen Index sind hier Iterationswerte z und Distanz/Abstand-Schätzung gültig.
                if (distanzschaetzer.bestimmt(charakteristik, igel)) {
                    return;
                }
                if (z.absq < domabsq) {
                    if (wukleusfinder.enthalten(charakteristik, igel)) {
                        return;
                    }
                    domabsq = z.absq;
                    domper = igel;
                } else if (z.periodisch() || innendetektor.innen()) {
                    charakteristik.setze(abstandschaetzer.kern(domper), innendetektor.mulq());
                    return;
                }
                distanzschaetzer.aktualisiere();
            } while (igel < hase);
            hase <<= 1;
        } while (igel < iterationslimit);
        charakteristik.setze(Charakteristik.NU_UNBESTIMMT, Charakteristik.LQ_UNBESTIMMT, innendetektor.mulq());
    }

    /**
     * z[0] = 0, z[n+1] = (z[n])^2 + c
     *
     * @return Das Maximum der benötigten Iterationen für den einzelnen Punkt.
     */
    int bestimme(final int p, final int itmax) {
        final var charakteristik = charakteristiken[p];
        assert charakteristik.offen();
        z.parametrisiere(a(p % m), b(p / m));
        z.starte1();
        if (koerperkopferkennung && z.trivial()) {
            charakteristik.setze(Charakteristik.koerperkopf(z.a), Charakteristik.LQ_UNBESTIMMT,
                    Charakteristik.MULQ_UNBESTIMMT);
        } else {
            bestimme(charakteristik);
            if (multiplikatoren) {
                charakteristik.koerperkopf(z);
            }
        }
        assert !charakteristik.offen() && charakteristik.bestimmt();
        final var it = itzahl(charakteristik.nu);
        return itmax(it, itmax);
    }

    int bestimme(final int p) {
        return bestimme(p, 0);
    }

    void initialisiere(final Konfiguration konfiguration) {
    }

}
