/******************************************************************************
 ** $Id: Ausschnitt.java 3621 2024-12-24 12:10:35Z wmh $
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package ebflmaennle.berechnung;

import java.awt.Dimension;
import java.awt.Point;
import java.awt.geom.Point2D;

public final class Ausschnitt {

    public static final int M_NORM = 18 * 64 + 1; // ~~ N_NORM * sqrt(2)
    public static final int N_NORM = 13 * 64 + 1;
    public static final int M_MIN = M_NORM / 2 + 1;
    public static final int N_MIN = N_NORM / 2 + 1;
    static final float G = 2.75f; // Breite
    static final float H = 2.5f; // Höhe
    /** Magnitudenschritteweite. */
    public static final int MAGNITUDENSCHRITTWEITE = 8;
    public static final float BASISPXS = Math.max(G / M_NORM, H / N_NORM);

    private static final double EPSDIV = 16;
    private static final double AM = -0.75;
    private static final double A0 = AM - BASISPXS * M_NORM / 2;
    private static final double BM = 0;
    private static final double B0 = BM - BASISPXS * N_NORM / 2;
    private static final int MAGNITUDENMINIMUM = -4 * MAGNITUDENSCHRITTWEITE;
    /**
     * Maximale Magnitude; berechnet sich aus der Bedingung GRUNDLAENGE/2^(mag/s) > 2^F*eps, was -mag/s*log(2) >
     * log(2^F*eps/GRUNDLAENGE) bedeutet und schlussendlich zu mag < -log(2^F*eps/GRUNDLAENGE)/log(2)*s =
     * (-log(eps/GRUNDLAENGE)/log(2)-F)s führt; hierbei sind s die Magnitudenschrittweite und F=1 ein fest gewählter
     * Wert.
     */
    private static final int MAGNITUDENMAXIMUM = (int) ((-Math.log(Math.ulp(1D) / BASISPXS) / Math.log(2) - 1)
            * MAGNITUDENSCHRITTWEITE);
    private static final int ANTIALIASING = 1; // Erlaubt: 1 oder 2
    private static final int VORSCHAU = 1; // Erlaubt: 1 oder 2
    private static final int DETAILLIERUNGSGRAD = 4;
    private static final int ITERATIONSLIMITSCHRANKE = 2 * Algorithmus.ITERATIONSLIMITMAXIMUM;
    private static final int ITERATIONSLIMITMINIMUM = 1 << 6;
    private static final int ITERATIONSLIMITVORGABE = 1 << 17;

    public static float skalierung(final int stufe) {
        return (float) Math.pow(2, (float) -stufe / (float) MAGNITUDENSCHRITTWEITE);
    }

    public static int zoomschritt(final int staerke) {
        return -staerke * MAGNITUDENSCHRITTWEITE;
    }

    private int antialiasing = ANTIALIASING;
    private int vorschau = VORSCHAU;
    private double am = AM;
    private double a0 = A0;
    private double bm = BM;
    private double b0 = B0;
    private float pxs = normzellenlaenge();
    private int magnitude = 0;
    private int m = M_NORM;
    private int n = N_NORM;
    private int iterationslimit = ITERATIONSLIMITVORGABE;

    public Dimension rechengroesse() {
        return new Dimension(m, n);
    }

    public Dimension leinwandgroesse() {
        return new Dimension(darstellungsgroesse(m), darstellungsgroesse(n));
    }

    public void detailliere() {
        iterationslimit = Integer.min(iterationslimit << 1, Algorithmus.ITERATIONSLIMITMAXIMUM);
    }

    public Point2D.Double ort(final int p) {
        return new Point2D.Double(a0 + p % m * pxs, b0 + p / m * pxs);
    }

    public int p(final Point punkt) {
        return realgroesse(punkt.x) + m * realgroesse(punkt.y);
    }

    public Point darstellungspunkt(final Point2D ort) {
        return new Point(darstellungsgroesse((ort.getX() - a0) / pxs), darstellungsgroesse((ort.getY() - b0) / pxs));
    }

    public Point darstellungsmittelpunkt() {
        return new Point(darstellungsgroesse(m / 2), darstellungsgroesse(n / 2));
    }

    public Point darstellungspunkt(final double relx, final double rely) {
        return new Point(darstellungsgroesse(relx * m), darstellungsgroesse(rely * n));
    }

    public void setze(final Dimension dimension) {
        final var w = realgroesse(dimension.width);
        m = w + w % vorschau;
        final var h = realgroesse(dimension.height);
        n = h + h % vorschau;
        a0 = am - pxs * m / 2;
        b0 = bm - pxs * n / 2;
    }

    public boolean skaliere(final int radstufe) {
        return skaliere(m / 2, n / 2, radstufe);
    }

    public boolean skaliere(final Point fixpunkt, final int radstufe) {
        final var i = realgroesse(fixpunkt.x);
        final var j = realgroesse(fixpunkt.y);
        return skaliere(i, j, radstufe);
    }

    public boolean fokussiere(final Point fixpunkt, final int zoomschritt) {
        return skaliere(fixpunkt, zoomschritt);
    }

    public boolean fokussiere(final double relx, final double rely, final int zoomschritt) {
        final var i = (int) (relx * m);
        final var j = (int) (rely * n);
        return skaliere(i, j, zoomschritt);
    }

    public void skizziere() {
        iterationslimit = Integer.max(iterationslimit >> 1, ITERATIONSLIMITMINIMUM);
    }

    public void verschiebe(final double relx, final double rely) {
        final var da = relx * m * pxs;
        a0 -= da;
        am -= da;
        final var db = rely * n * pxs;
        b0 -= db;
        bm -= db;
    }

    public void verschiebe(final Point delta) {
        final var i = realgroesse(delta.getX());
        a0 -= i * pxs;
        am -= i * pxs;
        final var j = realgroesse(delta.getY());
        b0 -= j * pxs;
        bm -= j * pxs;
    }

    public void aktiviereAntialiasing(final boolean aktiviert) {
        final var altaktiviert = antialiasing == 2;
        if (aktiviert != altaktiviert) {
            if (aktiviert) {
                antialiasing = 2;
                m *= 2;
                n *= 2;
                pxs = pxs / 2;
            } else {
                antialiasing = 1;
                m /= 2;
                n /= 2;
                pxs = 2 * pxs;
            }
        }
    }

    public void aktiviereVorschau(final boolean aktiviert) {
        final var altaktiviert = vorschau == 2;
        if (aktiviert != altaktiviert) {
            if (aktiviert) {
                vorschau = 2;
                m /= 2;
                n /= 2;
                pxs = 2 * pxs;
            } else {
                vorschau = 1;
                m *= 2;
                n *= 2;
                pxs = pxs / 2;
            }
        }
    }

    public int antialiasing() {
        return antialiasing;
    }

    public int vorschau() {
        return vorschau;
    }

    private float normzellenlaenge() {
        return BASISPXS * vorschau / antialiasing;
    }

    private int darstellungsgroesse(final double realgroesse) {
        return (int) (realgroesse * vorschau / antialiasing);
    }

    private int realgroesse(final int darstellungsgroesse) {
        return darstellungsgroesse * antialiasing / vorschau;
    }

    private double realgroesse(final double darstellungsgroesse) {
        return darstellungsgroesse * antialiasing / vorschau;
    }

    private double epsilon() {
        return Math.sqrt(pxs) / EPSDIV * vorschau / antialiasing;
    }

    private boolean fuellbar() {
        return m * pxs < 4 * G && n * pxs < 4 * H;
    }

    private boolean skaliere(final int i, final int j, final int stufe) {
        final var neumagnitude = Integer.max(Integer.min(magnitude - stufe, MAGNITUDENMAXIMUM), MAGNITUDENMINIMUM);
        final var skaliert = magnitude - stufe == neumagnitude;
        if (!skaliert) {
            return false;
        }
        magnitude = neumagnitude;
        final var laenge = pxs;
        pxs = normzellenlaenge() * skalierung(magnitude);
        a0 += (laenge - pxs) * i;
        am = a0 + pxs / 2 * m;
        b0 += (laenge - pxs) * j;
        bm = b0 + pxs / 2 * n;
        return true;
    }

    void adaptiere(final Option.Qualitaet qualitaet, final int istiterationsmaximum) {
        if (qualitaet == Option.Qualitaet.AUTOMATIK || qualitaet == Option.Qualitaet.OPTIMUM) {
            while (iterationslimit < Algorithmus.ITERATIONSLIMITMAXIMUM
                    && iterationslimit - iterationslimit / DETAILLIERUNGSGRAD < istiterationsmaximum) {
                detailliere();
            }
            while (iterationslimit > ITERATIONSLIMITMINIMUM
                    && iterationslimit / 2 - iterationslimit / DETAILLIERUNGSGRAD >= istiterationsmaximum) {
                skizziere();
            }
        }
    }

    Konfiguration konfiguration(final Option option, final boolean beschleunigt) {
        return new Konfiguration(option, new Dimension(m, n), iterationslimit, ITERATIONSLIMITSCHRANKE, fuellbar(),
                new Point2D.Double(a0, b0), epsilon(), pxs, antialiasing, vorschau, beschleunigt);
    }

}
