/******************************************************************************
 ** $Id: Algorithmus.java 3632 2024-12-24 16:37:11Z wmh $
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package ebflmaennle.berechnung;

import java.beans.PropertyChangeSupport;
import java.util.Arrays;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.RecursiveAction;
import java.util.concurrent.RecursiveTask;
import java.util.concurrent.atomic.AtomicInteger;

import ebflmaennle.berechnung.Kalkulator.Parameter;
import ebflmaennle.oberflaeche.Leinwand;

public final class Algorithmus extends PropertyChangeSupport {

    final record Schritte(int sx, int sy, int s) {
    }

    abstract public class Block {
        /** Anzahl der inneren Gitterpunkte in der x-Richtung (Realteil). */
        protected short dk;
        /** Anzahl der inneren Gitterpunkte in der y-Richtung (Imaginärteil). */
        protected short dl;
        /** Links oben. */
        protected int lo;
        /** Rechts oben. */
        protected int ro;
        /** Links unten. */
        protected int lu;
        /** Rechts unten. */
        protected int ru;

        // links mittig
        static int lm(final int lo, final int dl, final int m) {
            return lo + dl / 2 * m;
        }

        // rechts mittig
        static int rm(final int lo, final int dk, final int dl, final int m) {
            return lm(lo, dl, m) + dk;
        }

        // mittig oben
        static int mo(final int lo, final int dk) {
            return lo + dk / 2;
        }

        // mittig unten
        static int mu(final int lu, final int dk) {
            return lu + dk / 2;
        }

        static int mm(final int lm, final int dk) {
            return lm + dk / 2;
        }

        public short dk() {
            return dk;
        }

        public short dl() {
            return dl;
        }

        public int ecke() {
            return lo;
        }

        // links mittig
        int lm() {
            return lm(lo, dl, m);
        }

        // rechts mittig
        int rm() {
            return rm(lo, dk, dl, m);
        }

        // mittig oben
        int mo() {
            return mo(lo, dk);
        }

        // mittig unten
        int mu() {
            return mu(lu, dk);
        }

        // mitte mitte
        int mm() {
            return mm(lm(), dk);
        }

        /**
         * @param kontext
         * @return Für jeden Punkt wird eine bestimme Anzahl von Iterationen benötigt, um alle gewünschten Eigenschaften
         *         zu kennen. Hier wird das Maximum über diese Anzahlen zurückgeliefert.
         */
        abstract int bearbeite(final Kontext kontext);
    }

    private class Basisebene {

        private class Block extends Algorithmus.Block {

            private boolean aussen() {
                return charakteristiken[lo].aussen() || charakteristiken[ro].aussen() || charakteristiken[lu].aussen()
                        || charakteristiken[ru].aussen();
            }

            private float lq(final float nu) {
                if (nu > 0) {
                    return charakteristiken[lo].lq + charakteristiken[ro].lq + charakteristiken[lu].lq
                            + charakteristiken[ru].lq;
                }
                return Math.min(
                        Math.min(Math.min(charakteristiken[lo].lq, charakteristiken[ro].lq), charakteristiken[lu].lq),
                        charakteristiken[ru].lq);
            }

            private float aussennu() {
                final double min = innennu();
                final var potenzialmittel = 2 + min - Math.log(Math.pow(2, min - charakteristiken[lo].nu)
                        + Math.pow(2, min - charakteristiken[ro].nu) + Math.pow(2, min - charakteristiken[lu].nu)
                        + Math.pow(2, min - charakteristiken[ru].nu)) / Math.log(2);
                final float nu;
                if (konfiguration.aussenblockschaetzung) {
                    nu = Math.round(potenzialmittel);
                } else {
                    nu = (float) potenzialmittel;
                }
                if (Charakteristik.offen(nu)) {
                    return 0;
                }
                return nu;
            }

            private float aussenfuellung() {
                final var nu = charakteristiken[ru].nu;
                var p = lo;
                for (; p < ro; p++) {
                    if (nu != charakteristiken[p].nu) {
                        return Charakteristik.NU_OFFEN;
                    }
                }
                for (; p < ru; p += m) {
                    if (nu != charakteristiken[p].nu) {
                        return Charakteristik.NU_OFFEN;
                    }
                }
                for (; p > lu; p--) {
                    if (nu != charakteristiken[p].nu) {
                        return Charakteristik.NU_OFFEN;
                    }
                }
                for (; p > lo; p -= m) {
                    if (nu != charakteristiken[p].nu) {
                        return Charakteristik.NU_OFFEN;
                    }
                }
                return aussennu();
            }

            private float innennu() {
                return Math.min(
                        Math.min(Math.min(charakteristiken[lo].nu, charakteristiken[ro].nu), charakteristiken[lu].nu),
                        charakteristiken[ru].nu);
            }

            private float innenfuellung() {
                var p = lo;
                for (; p < ro; p++) {
                    if (charakteristiken[p].nichtinnen()) {
                        return Charakteristik.NU_OFFEN;
                    }
                }
                for (; p < ru; p += m) {
                    if (charakteristiken[p].nichtinnen()) {
                        return Charakteristik.NU_OFFEN;
                    }
                }
                for (; p > lu; p--) {
                    if (charakteristiken[p].nichtinnen()) {
                        return Charakteristik.NU_OFFEN;
                    }
                }
                for (; p > lo; p -= m) {
                    if (charakteristiken[p].nichtinnen()) {
                        return Charakteristik.NU_OFFEN;
                    }
                }
                return innennu();
            }

            private float zwangsfuellung(final boolean innenschnittmenge) {
                if (!innenschnittmenge) {
                    return aussennu();
                }
                if (aussen()) {
                    return Charakteristik.NU_OFFEN;
                }
                return innennu();
            }

            private float normalfuellung(final boolean innenschnittmenge) {
                final var aussenblockschaetzung = konfiguration.aussenblockschaetzung;
                final var innenblockschaetzung = konfiguration.innenblockschaetzung;
                if (innenschnittmenge && !innenblockschaetzung || !innenschnittmenge && !aussenblockschaetzung) {
                    return Charakteristik.NU_OFFEN;
                }
                if (innenschnittmenge) {
                    return innenfuellung();
                }
                return aussenfuellung();
            }

            private float fuellung(final boolean fuellzwang, final boolean innenschnittmenge) {
                if (!konfiguration.fuellbar) {
                    return Charakteristik.NU_OFFEN;
                }
                if (fuellzwang) {
                    return zwangsfuellung(innenschnittmenge);
                }
                return normalfuellung(innenschnittmenge);
            }

            private void fuelle(final Konfiguration konfiguration, final boolean innenschnittmenge,
                    final float fuellung) {
                final float nu;
                final float lq;
                final float mulq;
                if (konfiguration.prinzipien) {
                    nu = Charakteristik.NU_OFFEN;
                    if (innenschnittmenge) {
                        lq = Charakteristik.LQ_UNBESTIMMT;
                    } else {
                        lq = Charakteristik.LQ_GEFLUECHTET;
                    }
                    mulq = Charakteristik.MULQ_OFFEN;
                } else {
                    if (fuellung > 0 && konfiguration.aussenblockschaetzung) {
                        nu = Math.round(fuellung);
                    } else {
                        nu = fuellung;
                    }
                    lq = lq(nu);
                    if ((nu <= 0) && (lq == 0)) {
                        mulq = Charakteristik.MULQ_ZENTRUM;
                    } else {
                        mulq = Charakteristik.MULQ_UNBESTIMMT;
                    }
                }
                assert dk >= MINIMALKANTENLAENGE && dl >= MINIMALKANTENLAENGE;
                final var p = lo + m;
                for (int pl = p + 1, pr = p + dk, d = 1; d < dl; d++, pl += m, pr += m) {
                    for (var q = pl; q < pr; q++) {
                        charakteristiken[q].setze(nu, lq, mulq);
                    }
                }
            }

            protected int komplettiere(final Linienrechner linienrechner) {
                final var itmax = linienrechner.komplettiere(lo, ro, lu);
                leinwand.vormerke(this);
                return itmax;
            }

            protected void verankere(final int lo, final int ro, final int lu, final int ru, final int dk,
                    final int dl) {
                this.lo = lo;
                this.ro = ro;
                this.lu = lu;
                this.ru = ru;
                this.dk = (short) dk;
                this.dl = (short) dl;
                assert ro - lo == ru - lu && lu - lo == ru - ro && dk >= MINIMALKANTENLAENGE
                        && dl >= MINIMALINNENLAENGE;
            }

            protected void nominiere() {
                blockvorrat[blockvorratsgroesse.getAndIncrement()] = this;
            }

            protected int zerteile(final Linienrechner linienrechner) {
                return komplettiere(linienrechner);
            }

            /**
             * @param konfiguration
             * @return Für jeden Punkt wird eine bestimme Anzahl von Iterationen benötigt, um alle gewünschten
             *         Eigenschaften zu kennen. Hier wird das Maximum über diese Anzahlen zurückgeliefert.
             */
            @Override
            int bearbeite(final Kontext kontext) {
                final var innenschnittmenge = (!charakteristiken[lo].aussen() || !charakteristiken[ru].aussen()
                        || !charakteristiken[ro].aussen() || !charakteristiken[lu].aussen());
                final var fuellung = fuellung(kontext.fuellzwang, innenschnittmenge);
                if (Charakteristik.offen(fuellung)) {
                    return zerteile(kontext.linienrechner);
                }
                fuelle(konfiguration, innenschnittmenge, fuellung);
                leinwand.vormerke(this);
                return 0;
            }

        }

        /** Erzeugung von Blöcken für die Ebene. */
        private final class Erzeuger extends RecursiveAction {
            private static final long serialVersionUID = 1770814152045884179L;
            private static final int INTERVAL = 16;

            protected final int i1;
            protected final int i2;

            Erzeuger(final int i1, final int i2) {
                this.i1 = i1;
                this.i2 = i2;
            }

            @Override
            protected void compute() {
                if (INTERVAL <= i2 - i1) {
                    final var mitte = (i1 + i2) / 2;
                    invokeAll(new Erzeuger(i1, mitte), new Erzeuger(mitte, i2));
                } else {
                    for (var i = i1; i < i2; i++) {
                        bloecke[i] = block(i);
                    }
                }
            }
        }

        private final class Parallelisierer extends RecursiveTask<Integer> {
            private static final long serialVersionUID = 507051490556457419L;
            private static final int INTERVAL = 2;
            private final int links;
            private final int rechts;

            Parallelisierer(final int links, final int rechts) {
                this.links = links;
                this.rechts = rechts;
            }

            @Override
            protected Integer compute() {
                int itmax;
                if (INTERVAL <= rechts - links) {
                    final var mitte = (links + rechts) / 2;
                    assert links < mitte && mitte < rechts;
                    final var pl = new Parallelisierer(links, mitte);
                    final var pr = new Parallelisierer(mitte, rechts);
                    pl.fork();
                    itmax = Integer.max(pr.compute(), pl.join());

                } else {
                    assert rechts - links == INTERVAL - 1;
                    final var kontext = kontext();
                    itmax = blockabarbeitung[links].bearbeite(kontext);
                    kontextvorrat.push(kontext);
                }
                return itmax;
            }
        }

        private final static int ABBRUCHSKALIERUNG = 2;

        /** Die Ebenentiefe, mögliche Werte von 0 (zuoberst) bis zum Maximum. */
        protected final int e;
        /** Anzahl der benötigten Blöcke der Ebene */
        protected int s;
        /** Alle Blöcke der Ebene in einem Feld. */
        protected final Block[] bloecke;
        /** Gibt an, ob Zerteilung vertikal erfolgen soll. */
        protected boolean vertikal;
        /** Gibt an, ob die Berechnung auf dieser Ebene abbrechbar ist. */
        protected boolean abbrechbar;

        Basisebene(final int e) {
            this.e = e;
            this.s = 1 << e; // Anzahl der benötigten Blöcke der Ebene
            this.bloecke = new Block[s];
            new Erzeuger(0, s).invoke();
        }

        protected Block block(final int i) {
            return new Block();
        }

        protected void verankere(final boolean vertikal) {
            this.vertikal = vertikal;
        }

        int bearbeite(final Konfiguration konfiguration, final int s) {
            final var innenquadratlaenge = Math.min(m, n);
            abbrechbar = innenquadratlaenge >> ((e + 1) / 2 + ABBRUCHSKALIERUNG) <= s;
            var itmax = 0;
            final var anzahl = blockvorratsgroesse.getAndSet(0);
            if (anzahl > 0) {
                final var austauschsmerker = blockabarbeitung;
                blockabarbeitung = blockvorrat;
                blockvorrat = austauschsmerker;
                itmax = new Parallelisierer(0, anzahl).invoke();
            }
            return itmax;
        }

        Kontext kontext() {
            Kontext kontext;
            while ((kontext = kontextvorrat.poll()) == null) {
                Thread.yield();
            }
            kontext.finalisiere(abgebrochen && abbrechbar);
            return kontext;
        }
    }

    private class Zwischenebene extends Basisebene {

        private final class Verankerer extends RecursiveAction {
            private static final long serialVersionUID = 1770814152045884179L;
            private static final int INTERVAL = 16;

            protected final int i1;
            protected final int i2;

            Verankerer(final int i1, final int i2) {
                this.i1 = i1;
                this.i2 = i2;
            }

            @Override
            protected void compute() {
                if (INTERVAL <= i2 - i1) {
                    final var mitte = (i1 + i2) / 2;
                    invokeAll(new Verankerer(i1, mitte), new Verankerer(mitte, i2));
                } else {
                    for (var i = i1; i < i2; i++) {
                        ((Block) bloecke[i]).verankere();
                    }
                }
            }
        }

        class Block extends Basisebene.Block {

            protected final Basisebene.Block b1; // Erster Verfeinerungsblock
            protected final Basisebene.Block b2; // Zweiter Verfeinerungsblock

            Block(final Basisebene.Block b1, final Basisebene.Block b2) {
                this.b1 = b1;
                this.b2 = b2;
            }

            private boolean zerteilbar() {
                return dk > 2 * MINIMALKANTENLAENGE && dl > 2 * MINIMALKANTENLAENGE;
            }

            protected void verankere() {
                if (!zerteilbar()) {
                    return;
                }
                if (vertikal) {
                    final var dk2 = dk / 2;
                    final var mo = lo + dk2; // mittig oben
                    final var mu = lu + dk2; // mittig unten
                    b1.verankere(lo, mo, lu, mu, dk2, dl);
                    b2.verankere(mo, ro, mu, ru, dk - dk2, dl);
                } else {
                    final var dl2 = dl / 2;
                    final var lm = lo + dl2 * m; // links mittig
                    final var rm = lm + dk; // rechts mittig
                    b1.verankere(lo, ro, lm, rm, dk, dl2);
                    b2.verankere(lm, rm, lu, ru, dk, dl - dl2);
                }
            }

            protected int schneideHorizontal(final Linienrechner linienrechner, final int l, final int r) {
                return linienrechner.zeile().bestimme(l, r);
            }

            protected int schneideVertikal(final Linienrechner linienrechner, final int o, final int u) {
                return linienrechner.spalte().bestimme(o, u);
            }

            @Override
            protected int zerteile(final Linienrechner linienrechner) {
                if (!zerteilbar()) {
                    return komplettiere(linienrechner);
                }
                assert 2 <= ro - lo && 2 * m <= lu - lo && 2 * m <= ru - ro && !charakteristiken[lo].offen()
                        && !charakteristiken[ro].offen() && !charakteristiken[lu].offen()
                        && !charakteristiken[ru].offen();
                final int itmax;
                if (vertikal) {
                    itmax = schneideVertikal(linienrechner, b1.ro, b1.ru);
                } else {
                    itmax = schneideHorizontal(linienrechner, b1.lu, b1.ru);
                }
                b1.nominiere();
                b2.nominiere();
                return itmax;
            }

        }

        Zwischenebene(final int e) {
            super(e);
        }

        @Override
        protected Block block(final int i) {
            final var bloecke = ebenen[e + 1].bloecke;
            return new Block(bloecke[2 * i], bloecke[2 * i + 1]);
        }

        @Override
        protected void verankere(final boolean vertikal) {
            super.verankere(vertikal);
            new Verankerer(0, s).invoke();
        }

    }

    private class Startebene extends Zwischenebene {

        private final class ParallelisiererHorizontal extends RecursiveTask<Integer> {
            private static final long serialVersionUID = -7374098759567173379L;
            private final int links;
            private final int rechts;

            ParallelisiererHorizontal(final int links, final int rechts) {
                this.links = links;
                this.rechts = rechts;
            }

            @Override
            public Integer compute() {
                if (m < 2 * (rechts - links)) {
                    final var mitte = (rechts + links) / 2;
                    final var hpl = new ParallelisiererHorizontal(links, mitte);
                    final var hpr = new ParallelisiererHorizontal(mitte, rechts);
                    hpl.fork();
                    return Integer.max(hpr.compute(), hpl.join());
                }
                final var kontext = kontext();
                final var max = kontext.linienrechner.zeile().bestimme(links, rechts);
                kontextvorrat.push(kontext);
                return max;
            }
        }

        private final class ParallelisiererVertikal extends RecursiveTask<Integer> {
            private static final long serialVersionUID = -8260785119505095168L;
            private final int oben;
            private final int unten;

            ParallelisiererVertikal(final int oben, final int unten) {
                this.oben = oben;
                this.unten = unten;
            }

            @Override
            public Integer compute() {
                if (mn < 2 * (unten - oben)) {
                    final var mitte = oben + (unten - oben) / m / 2 * m;
                    final var vpo = new ParallelisiererVertikal(oben, mitte);
                    final var vpu = new ParallelisiererVertikal(mitte, unten);
                    vpo.fork();
                    return Integer.max(vpu.compute(), vpo.join());
                }
                final var kontext = kontext();
                final var max = kontext.linienrechner.spalte().bestimme(oben, unten);
                kontextvorrat.push(kontext);
                return max;
            }
        }

        class Block extends Zwischenebene.Block {

            Block(final Basisebene.Block b1, final Basisebene.Block b2) {
                super(ebenen[1].bloecke[0], ebenen[1].bloecke[1]);
            }

            @Override
            protected int schneideHorizontal(final Linienrechner linienrechner, final int l, final int r) {
                return new ParallelisiererHorizontal(l, r).compute();
            }

            @Override
            protected int schneideVertikal(final Linienrechner linienrechner, final int o, final int u) {
                return new ParallelisiererVertikal(o, u).compute();
            }
        }

        Startebene() {
            super(0);
        }

        private int umrahme() {
            final var block = bloecke[0];
            final var lo = block.lo;
            final var mo = block.mo();
            final var ro = block.ro;
            final var lm = block.lm();
            final var mm = block.mm();
            final var rm = block.rm();
            final var lu = block.lu;
            final var mu = block.mu();
            final var ru = block.ru;
            final var stuetzpunkte = Arrays.asList(lo, mo, ro, lm, mm, rm, lu, mu, ru);
            var itmax = stuetzpunkte.parallelStream().mapToInt(p -> {
                final var kontext = kontext();
                final var itzahl = kontext.charakterisierung.bestimme(p);
                kontextvorrat.push(kontext);
                return itzahl;
            }).max().getAsInt();
            final var oben = new ParallelisiererHorizontal(lo, ro).fork();
            final var links = new ParallelisiererVertikal(lo, lu).fork();
            final var rechts = new ParallelisiererVertikal(ro, ru).fork();
            itmax = Integer.max(new ParallelisiererHorizontal(lu, ru).compute(), itmax);
            itmax = Integer.max(oben.join(), itmax);
            itmax = Integer.max(links.join(), itmax);
            return Integer.max(rechts.join(), itmax);
        }

        @Override
        protected Block block(final int i) {
            final var bloecke = ebenen[e + 1].bloecke;
            return new Block(bloecke[2 * i], bloecke[2 * i + 1]);
        }

        void verankere() {
            bloecke[0].verankere(0, m - 1, mn - m, mn - 1, m - 1, n - 1);
        }

        @Override
        int bearbeite(final Konfiguration konfiguration, final int s) {
            final var block = bloecke[0];
            var itmax = umrahme();
            blockvorratsgroesse.set(0);
            final var kontext = kontext();
            itmax = Integer.max(itmax, block.bearbeite(kontext));
            kontextvorrat.push(kontext);
            return itmax;
        }
    }

    private static final long serialVersionUID = 4591855844064083831L;

    private static final double DIV = 4;
    private static final int EBENENANZAHL = 19;
    private static final int EBENENBLOCKMAXIMALANZAHL = 1 << (EBENENANZAHL - 1);
    private static final int OOP_FELD = 16;
    private static final int OOP_INSTANZ = 4;
    static final int ITERATIONSLIMITMAXIMUM = 1 << (Integer.SIZE - 5);
    /** Minimale Anzahl von inneren Gitterlinien für Blöcke, die noch nominierbar sind. */
    static final int MINIMALINNENLAENGE = 3;
    /** Minimale Anzahl von Gitterlinien der Kantenlängen für Blöcke, die noch nominierbar sind. */
    static final int MINIMALKANTENLAENGE = MINIMALINNENLAENGE + 1;

    public static long speicherdatenbedarf() {
        // Speicherplatzbedarf der Felder, welche die Blöcke halten (Vorrat und Abarbeitung)
        final long sblockschlangen = 2 * (OOP_FELD + OOP_INSTANZ * EBENENBLOCKMAXIMALANZAHL);
        // Speicherplatzbedarf der Felder, welche die Blöcke halten (Ebenen 0..*)
        final long sblockebenen = OOP_FELD * EBENENANZAHL + OOP_INSTANZ * 2 * EBENENBLOCKMAXIMALANZAHL;
        // Größe eines Zwischenblocks
        final var szwischenblock = 56;
        // Größe eines Basisblocks
        final var sbasisblock = 40;
        // Speicherbedarf aller Blöcke
        final long sbloecke = (szwischenblock + sbasisblock) * EBENENBLOCKMAXIMALANZAHL - szwischenblock;
        // Gesamtspeicherbedarf
        return sblockschlangen + sblockebenen + sbloecke;
    }

    private final AtomicInteger blockvorratsgroesse = new AtomicInteger();
    private final ConcurrentLinkedDeque<Kontext> kontextvorrat = new ConcurrentLinkedDeque<>();
    private final Basisebene[] ebenen = new Basisebene[EBENENANZAHL];
    private final Charakteristik[] charakteristiken;
    private final Leinwand leinwand;

    private int m;
    private int n;
    private int mn;
    private Schritte schritte;
    private Konfiguration konfiguration;
    /** Die Blöcke zur Bearbeitung in der zur Berechnung aktuell anstehenden Ebene. */
    private Block[] blockabarbeitung = new Block[EBENENBLOCKMAXIMALANZAHL];
    /** Die Blöcke zur Bearbeitung in der zur Berechnung kommenden anstehenden Ebene. */
    private Block[] blockvorrat = new Block[EBENENBLOCKMAXIMALANZAHL];
    /** Gibt an, ob der Algorithmus vorzeitig abgebrochen werden soll. */
    private volatile boolean abgebrochen;
    /** Gibt an, ob der Algorithmus einem interaktiven Verhalten den Vorzug geben soll. */
    private boolean interaktiv;

    Algorithmus(final Charakteristik[] charakteristiken, final Leinwand leinwand) {
        super(new Object());
        this.charakteristiken = charakteristiken;
        this.leinwand = leinwand;
        var e = ebenen.length - 1;
        ebenen[e] = new Basisebene(e);
        while (e > 1) {
            e--;
            ebenen[e] = new Zwischenebene(e);
        }
        ebenen[0] = new Startebene();
    }

    void abbreche() {
        for (final var kontext : kontextvorrat) {
            kontext.linienrechner.waehle(interaktiv);
        }
        abgebrochen = true;
    }

    void verankere(final int m, final int n, final int antialiasing, final int vorschau) {
        this.m = m;
        this.n = n;
        this.mn = m * n;
        var vertikal = m > n;
        ((Startebene) ebenen[0]).verankere();
        for (final var ebene : ebenen) {
            ebene.verankere(vertikal);
            vertikal = !vertikal;
        }
        var dm = m - 1;
        var dn = n - 1;
        var vertikal1 = dm > dn;
        while (dm > 2 * MINIMALKANTENLAENGE && dn > 2 * MINIMALKANTENLAENGE) {
            if (vertikal1) {
                dm /= 2;
            } else {
                dn /= 2;
            }
            vertikal1 = !vertikal1;
        }
        dm = dm / 2 + 1;
        dn = dn / 2 + 1;
        final var s = Math.min(dm, dn) * antialiasing / vorschau;
        schritte = new Schritte(dm, m * dn, s);
    }

    int bearbeite(final Konfiguration konfiguration) {
        this.konfiguration = konfiguration;
        kontextvorrat.clear();
        for (var i = 0; i <= 4 * ForkJoinPool.getCommonPoolParallelism(); i++) {
            final var delta = konfiguration.epsilon / DIV;
            kontextvorrat.push(new Kontext(charakteristiken, konfiguration, m, delta, schritte));
        }
        abgebrochen = false;
        interaktiv = konfiguration.bildfrequenz;
        var itmax = 0;
        for (final var ebene : ebenen) {
            /* e = ebene.e ungerade: horizontal und vertikal gleich viele Unterteilungen 2^((e+1)/2) */
            for (final var kontext : kontextvorrat) {
                kontext.linienrechner.kippe(ebene.vertikal);
            }
            itmax = Integer.max(ebene.bearbeite(konfiguration, schritte.s), itmax);
        }
        return itmax;
    }

    public N nukleus(final Parameter parameter) {
        final var nu = parameter.charakteristik().nu;
        if (!Charakteristik.innen(nu)) {
            return null;
        }
        final var kontext = ebenen[0].kontext();
        final var periode = -(int) nu;
        final var n = kontext.nukleusfinder.finde(parameter.ort().getX(), parameter.ort().getY(), periode);
        kontextvorrat.push(kontext);
        return n;
    }

}
