/******************************************************************************
 ** $Id: Abstandschaetzer.java 3631 2024-12-24 16:34:32Z wmh $
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package ebflmaennle.berechnung;

import ebflmaennle.berechnung.Charakteristik.Kern;

abstract class Abstandschaetzer {

    private static class Platzhalter extends Abstandschaetzer {

        public Platzhalter(final Z z, final double pxs) {
            super(pxs);
        }

        @Override
        protected Kern kern(final int periode) {
            return new Kern(-periode, schranke);
        }
    }

    private static class Abschaetzer extends Abstandschaetzer {

        protected final Z t;
        protected final Dc dc;
        protected final Dz dz;
        protected final W w;

        Abschaetzer(final W w, final double pxs) {
            super(pxs);
            this.w = w;
            t = w.t;
            dc = new Dc(t);
            dz = w.dz;
        }

        Abschaetzer(final N n) {
            super(0);
            w = null;
            t = n.t;
            dc = n.dc;
            dz = new Dz(t);
        }

        protected void iteriere(final double spur) {
            dc.iteriere(); // dc = 2 * t * dc + 1;
            dz.iteriere(spur); // dz = 2 * t * dz
            t.iteriere(); // t = t * t + c
        }

        @Override
        protected Kern kern(final int periode) {
            var dcdzx = 0d;
            var dcdzy = 0d;
            var dzdzx = 2d;
            var dzdzy = 0d;
            w.uebertrage();
            dc.starte1();
            var p = 1;
            var vielfach = periode % 2 == 0;
            var diskriminante = 1 - dz.absq;
            if (!vielfach || diskriminante < 0 || w.divergent()) {
                p++;
                var spur = t.x + t.y;
                // dcdz = 2 * (t * dcdz + dz * dc)
                dcdzx = 2 * (dz.x * dc.x - dz.y * dc.y);
                dcdzy = 2 * (dz.x * dc.y + dz.y * dc.x);
                // dzdz = 2 * (dz * dz + t * dzdz)
                dzdzx = 2 * (2 * t.x + dz.xq - dz.yq);
                dzdzy = 2 * (2 * t.y + 2 * dz.x * dz.y);
                iteriere(spur);
                diskriminante = 1 - dz.absq; // Kriterium für Inneres
                if (periode > 2) {
                    p++;
                    while ((!(vielfach = periode % p == 0) || diskriminante < 0 || w.divergent())) {
                        spur = t.x + t.y;
                        // dcdz = 2 * (t * dcdz + dz * dc)
                        final var kdcdz1 = dcdzx * spur;
                        final var kdcdz2 = t.y * (dcdzx + dcdzy);
                        final var kdcdz3 = t.x * (dcdzy - dcdzx);
                        dcdzx = 2 * (kdcdz1 - kdcdz2 + dz.x * dc.x - dz.y * dc.y);
                        dcdzy = 2 * (kdcdz1 + kdcdz3 + dz.x * dc.y + dz.y * dc.x);
                        // dzdz = 2 * (dz * dz + t * dzdz)
                        final var kdzdz1 = dzdzx * spur;
                        final var kdzdz2 = t.y * (dzdzx + dzdzy);
                        final var kdzdz3 = t.x * (dzdzy - dzdzx);
                        dzdzx = 2 * (kdzdz1 - kdzdz2 + dz.xq - dz.yq);
                        dzdzy = 2 * (kdzdz1 + kdzdz3 + 2 * dz.x * dz.y);
                        iteriere(spur);
                        diskriminante = 1 - dz.absq; // Kriterium für Inneres
                        if (p == periode) {
                            break;
                        }
                        p++;
                    }
                }
            }
            final double lq;
            if (vielfach && diskriminante >= 0) {
                assert t.innenkandidat();
                // abstand = (1 - |dz|²) / |dcdz + dzdz * dc / (1 - dz)|
                final var hx = dzdzx * dc.x - dzdzy * dc.y;
                final var hy = dzdzy * dc.x + dzdzx * dc.y;
                final var dzx1 = 1 - dz.x;
                final var absqdiv = dzx1 * dzx1 + dz.yq;
                final var hhx = dcdzx + (hx * dzx1 - hy * dz.y) / absqdiv;
                final var hhy = dcdzy + (hx * dz.y + hy * dzx1) / absqdiv;
                lq = -diskriminante * diskriminante / (hhx * hhx + hhy * hhy);
            } else {
                lq = -Charakteristik.LQ_GEFLUECHTET; // War eingangs fälschlicherweise der Mandelbrotmenge zugeschlagen.
            }
            return new Kern(-p, (float) lq);
        }

    }

    public static class Groessenschaetzer extends Abschaetzer {
        private final N n;

        Groessenschaetzer(final N n) {
            super(n);
            this.n = n;
        }

        @Override
        protected Kern kern(final int domper) {
            var dzdzx = 2d;
            var dzdzy = 0d;
            n.uebertrage();
            dz.beginne(); // dz[1]
            int p;
            for (p = 1; p < domper; p++) {
                final var spur = t.x + t.y;
                // dzdz = 2 * (dz * dz + t * dzdz)
                final var kdzdz1 = dzdzx * spur;
                final var kdzdz2 = t.y * (dzdzx + dzdzy);
                final var kdzdz3 = t.x * (dzdzy - dzdzx);
                dzdzx = 2 * (kdzdz1 - kdzdz2 + dz.xq - dz.yq);
                dzdzy = 2 * (kdzdz1 + kdzdz3 + 2 * dz.x * dz.y);
                iteriere(spur);
            }
            assert t.innenkandidat();
            // abstand = 1 / |dzdz * dc|
            return new Kern(-domper, (float) -(1 / (dzdzx * dzdzx + dzdzy * dzdzy) / (dc.x * dc.x + dc.y * dc.y)));
        }

    }

    static Abstandschaetzer erzeuge(final W w, final double pxs, final double epsilon,
            final boolean abstandschaetzung) {
        final Abstandschaetzer abstandschaetzer;
        if (abstandschaetzung) {
            abstandschaetzer = new Abschaetzer(w, pxs);
        } else {
            abstandschaetzer = new Platzhalter(w.z, pxs);
        }
        return abstandschaetzer;
    }

    protected final float schranke;

    private Abstandschaetzer(final double pxs) {
        schranke = -(float) (pxs * pxs);
    }

    /** Liefert Schätzwert für den quadrierten Innenabstand und korrigierte Periode. */
    protected abstract Kern kern(final int domper);

}
