/******************************************************************************
 ** $Id: Main.java 3595 2024-09-07 18:39:14Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Programms Ebflmännle.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package ebflmaennle;

import java.awt.Dimension;
import java.awt.EventQueue;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.text.ParseException;

import javax.swing.Action;
import javax.swing.JOptionPane;

import ebflmaennle.berechnung.Algorithmus;
import ebflmaennle.berechnung.Ausschnitt;
import ebflmaennle.berechnung.Kalkulator;
import ebflmaennle.berechnung.Karte;
import ebflmaennle.berechnung.Option;
import ebflmaennle.farbe.Modell;
import ebflmaennle.farbe.Parameter;
import ebflmaennle.oberflaeche.Aktion;
import ebflmaennle.oberflaeche.Aktionen;
import ebflmaennle.oberflaeche.Definitionseingabe;
import ebflmaennle.oberflaeche.Definitionsexport;
import ebflmaennle.oberflaeche.Definitionsimport;
import ebflmaennle.oberflaeche.Definitionsoperation;
import ebflmaennle.oberflaeche.Delegation;
import ebflmaennle.oberflaeche.Fenster;
import ebflmaennle.oberflaeche.Foto;
import ebflmaennle.oberflaeche.Gestaltung;
import ebflmaennle.oberflaeche.Info;
import ebflmaennle.oberflaeche.Inspektor;
import ebflmaennle.oberflaeche.Leinwand;
import ebflmaennle.oberflaeche.Menue;
import ebflmaennle.oberflaeche.Tafel;
import ebflmaennle.oberflaeche.Verfolgung;

public final class Main {

    public enum Betrieb {
        MINIMAL, EINGESCHRAENKT, VOLLSTAENDIG
    }

    public record Definition(Option option, Parameter parameter, Ausschnitt ausschnitt) {
    }

    public static final int KERNANZAHL = Runtime.getRuntime().availableProcessors();

    private static final Dimension MINIMUM = new Dimension(Ausschnitt.M_MIN, Ausschnitt.N_MIN);
    private static final Dimension STANDARD = Toolkit.getDefaultToolkit().getScreenSize();
    private static final long SPEICHERARBEITSBEDARF = 20_000_000;

    public static void main(final String[] args) {
        final var betrieb = betrieb();
        if (betrieb == Betrieb.MINIMAL) {
            JOptionPane.showMessageDialog(null,
                    "Arbeitsspeicher ist sehr knapp, daher ist nur ein stark eingeschränkter Betrieb möglich.",
                    "Ebflmännle: Warnung", JOptionPane.WARNING_MESSAGE);
        }
        EventQueue.invokeLater(() -> {
            try {
                starte(betrieb);
            } catch (final Exception e) {
                e.printStackTrace();
            }
        });
    }

    public static long speicherbedarf(final int punktanzahl) {
        return Karte.speicherdatenbedarf(punktanzahl) + Algorithmus.speicherdatenbedarf()
                + Leinwand.speicherdatenbedarf(punktanzahl);
    }

    private static Betrieb betrieb() {
        final var runtime = Runtime.getRuntime();
        final var speichervorrat = runtime.maxMemory() - runtime.totalMemory() + runtime.freeMemory();
        if (speicherbedarf(2 * STANDARD.width * 2 * STANDARD.height) + 2 * SPEICHERARBEITSBEDARF <= speichervorrat) {
            return Betrieb.VOLLSTAENDIG;
        }
        if (speicherbedarf(STANDARD.width * STANDARD.height) + SPEICHERARBEITSBEDARF <= speichervorrat) {
            return Betrieb.EINGESCHRAENKT;
        }
        return Betrieb.MINIMAL;
    }

    private static Dimension dimension(final Betrieb betrieb) {
        return switch (betrieb) {
            case VOLLSTAENDIG -> new Dimension(2 * STANDARD.width, 2 * STANDARD.height);
            case EINGESCHRAENKT -> STANDARD;
            default -> MINIMUM;
        };
    }

    private static Menue menue(final Betrieb betrieb, final Fenster fenster, final Kalkulator kalkulator,
            final Definition definition, final Modell farbmodellbeobachter, final Leinwand leinwand,
            final Definitionseingabe eingabe, final Definitionsexport ausfuhr, final Definitionsimport einfuhr,
            final Foto foto, final Gestaltung gestaltung, final Tafel tafel, final Inspektor pipette,
            final Inspektor zentrum, final Info info, final Aktion ende, final Verfolgung verfolgung) {

        final var delegation = new Delegation(kalkulator, definition.ausschnitt(), definition.option());
        final var ausschnittsbeobachter = delegation.ausschnittsbeobachter(leinwand);
        final var prinzipienaktion = delegation.prinzipienaktion();
        prinzipienaktion.addPropertyChangeListener(gestaltung);
        final var multiplikatorenaktion = delegation.multiplikatorensaktion();
        multiplikatorenaktion.addPropertyChangeListener(gestaltung);
        final var abstandschaetzungsaktion = delegation.abstandschaetzungsaktion();
        abstandschaetzungsaktion.addPropertyChangeListener(gestaltung);
        final var scheibenaktion = delegation.scheibenaktion();
        scheibenaktion.addPropertyChangeListener(leinwand);
        final var pipettenaktion = delegation.pipettenaktion(pipette);
        pipettenaktion.addPropertyChangeListener(verfolgung);
        final var zentrumaktion = delegation.zentrumaktion(zentrum);
        zentrumaktion.addPropertyChangeListener(verfolgung);
        final var zentrumdetektionsaktion = delegation.zentrumdetektionsaktion();
        zentrumdetektionsaktion.addPropertyChangeListener(gestaltung);
        final var andreaskreuzaktion = delegation.andreaskreuzaktion();
        andreaskreuzaktion.addPropertyChangeListener(leinwand);
        andreaskreuzaktion.addPropertyChangeListener(verfolgung);
        final var innenblockschaetzungsaktion = delegation.innenblockschaetzungsaktion();
        innenblockschaetzungsaktion.addPropertyChangeListener(gestaltung);
        final var abbruchaktion = new Aktion("Abbruch", KeyEvent.VK_ESCAPE, "Abbruch") {
            private static final long serialVersionUID = 5859662157178295051L;

            @Override
            public void actionPerformed(final ActionEvent event) {
                kalkulator.abbreche();
            }
        };
        abbruchaktion.setEnabled(false);

        pipette.registriere(pipettenaktion, verfolgung);
        zentrum.registriere(zentrumaktion, kalkulator);

        kalkulator.registriere(zentrumaktion, verfolgung);
        kalkulator.registriere(andreaskreuzaktion, verfolgung);

        kalkulator.addPropertyChangeListener("Berechnung",
                ereignis -> abbruchaktion.setEnabled(ereignis.getNewValue().equals(true)));

        final var eingabeaktion = new Aktion("Definitionseingabe...", KeyEvent.VK_E, InputEvent.CTRL_DOWN_MASK,
                "Eingabe") {
            private static final long serialVersionUID = 8040450750161740072L;

            @Override
            public void actionPerformed(final ActionEvent event) {
                eingabe.setVisible(true);
            }
        };
        eingabeaktion.setEnabled(false);
        final var ausfuhraktion = new Aktion("Definitionsexport...", KeyEvent.VK_X, InputEvent.CTRL_DOWN_MASK,
                "Export") {
            private static final long serialVersionUID = 8250283015527489629L;

            @Override
            public void actionPerformed(final ActionEvent event) {
                ausfuhr.vollfuehre();
            }
        };
        final var einfuhraktion = new Aktion("Definitionsimport...", KeyEvent.VK_I, InputEvent.CTRL_DOWN_MASK,
                "Import") {
            private static final long serialVersionUID = 795935819972788641L;

            @Override
            public void actionPerformed(final ActionEvent event) {
                try {
                    if (einfuhr.vollfuehre()) {
                        firePropertyChange("Import", null, definition);
                    }
                } catch (final ParseException | NoSuchFieldException | IllegalAccessException ausnahme) {
                    ausnahme.printStackTrace();
                }
            }
        };
        einfuhraktion.addPropertyChangeListener(leinwand);
        einfuhraktion.addPropertyChangeListener(gestaltung);
        einfuhraktion.addPropertyChangeListener(kalkulator);

        final var fotoaktion = new Aktion("Bildspeicherung...", KeyEvent.VK_S, InputEvent.CTRL_DOWN_MASK, "Foto") {
            private static final long serialVersionUID = -5954835747070585099L;

            @Override
            public void actionPerformed(final ActionEvent event) {
                foto.vollfuehre(leinwand.bild());
            }
        };

        final var farbgestaltungsaktion = new Aktion("Farbgestaltung", KeyEvent.VK_F) {
            private static final long serialVersionUID = 6539940166997263015L;

            @Override
            public void actionPerformed(final ActionEvent event) {
                gestaltung.setVisible((boolean) getValue(SELECTED_KEY));
            }
        };
        final var tafelaktion = new Aktion("Tafel", KeyEvent.VK_F9) {
            private static final long serialVersionUID = -6823139897706373780L;

            @Override
            public void actionPerformed(final ActionEvent event) {
                tafel.setVisible((boolean) getValue(SELECTED_KEY));
            }
        };
        final var infoaktion = new Aktion("<html>Über <i>Ebflmännle</i></html>", KeyEvent.VK_F1,
                InputEvent.CTRL_DOWN_MASK, "Info") {
            private static final long serialVersionUID = 4677453714157071674L;

            @Override
            public void actionPerformed(final ActionEvent event) {
                info.setVisible(true);
            }
        };

        gestaltung.addWindowListener(new WindowAdapter() {
            @Override
            public void windowClosing(final WindowEvent event) {
                farbgestaltungsaktion.putValue(Action.SELECTED_KEY, false);
            }
        });
        tafel.addWindowListener(new WindowAdapter() {
            @Override
            public void windowClosing(final WindowEvent event) {
                tafelaktion.putValue(Action.SELECTED_KEY, false);
            }
        });
        pipette.addWindowListener(new WindowAdapter() {
            @Override
            public void windowClosing(final WindowEvent event) {
                pipettenaktion.putValue(Action.SELECTED_KEY, false);
            }
        });

        return new Menue(betrieb, kalkulator, farbmodellbeobachter, gestaltung, delegation, ausschnittsbeobachter,
                new Aktionen(new Aktion[] {
                    eingabeaktion,
                    ausfuhraktion,
                    einfuhraktion,
                    fotoaktion,
                    farbgestaltungsaktion,
                    prinzipienaktion,
                    tafelaktion,
                    pipettenaktion,
                    zentrumaktion,
                    andreaskreuzaktion,
                    multiplikatorenaktion,
                    zentrumdetektionsaktion,
                    abstandschaetzungsaktion,
                    scheibenaktion,
                    infoaktion,
                    innenblockschaetzungsaktion,
                    abbruchaktion,
                    ende }));
    }

    private static void starte(final Betrieb betrieb) {
        final var karte = new Karte(dimension(betrieb));
        final var ausschnitt = new Ausschnitt();
        final var option = new Option();
        final var parameter = new Parameter();
        final var farbmodellbeobachter = new Modell(parameter);
        final var standardgroesse = betrieb != Betrieb.MINIMAL ? new Dimension(Ausschnitt.M_NORM, Ausschnitt.N_NORM)
                : new Dimension(Ausschnitt.M_MIN, Ausschnitt.N_MIN);
        final var maximalgroesse = betrieb != Betrieb.MINIMAL ? null
                : new Dimension(Ausschnitt.M_MIN, Ausschnitt.N_MIN);
        final var leinwand = new Leinwand(standardgroesse, maximalgroesse, farbmodellbeobachter,
                karte.charakteristiken);
        final var kalkulator = new Kalkulator(option, karte, ausschnitt, leinwand);
        final var verfolgung = new Verfolgung(ausschnitt, kalkulator, leinwand);
        final var ende = new Aktion("Beenden", KeyEvent.VK_F4, InputEvent.ALT_DOWN_MASK, "Ende") {
            private static final long serialVersionUID = 8878387920327474027L;

            @Override
            public void actionPerformed(final ActionEvent event) {
                final var wahl = JOptionPane.showConfirmDialog(null, "Programm beenden?", "Ebflmännle",
                        JOptionPane.YES_NO_OPTION);
                if (wahl == JOptionPane.YES_OPTION) {
                    System.exit(0);
                }
            }
        };
        final var fenster = new Fenster();
        fenster.addWindowListener(new WindowAdapter() {
            @Override
            public void windowClosing(final WindowEvent event) {
                ende.actionPerformed(new ActionEvent(this, ActionEvent.ACTION_PERFORMED, null));
            }
        });
        final var eingabe = new Definitionseingabe(fenster);
        final var definition = new Definition(option, parameter, ausschnitt);
        final var definitionsoperation = new Definitionsoperation(definition);
        final var ausfuhr = new Definitionsexport(fenster, definitionsoperation);
        final var einfuhr = new Definitionsimport(fenster, definitionsoperation);
        final var foto = new Foto(fenster);
        final var gestaltung = new Gestaltung(fenster, farbmodellbeobachter);
        final var tafel = new Tafel(fenster);
        final var zentrum = new Inspektor(fenster, "Ebflmännle: Inspektor (Zentrum)");
        final var pipette = new Inspektor(fenster, "Ebflmännle: Inspektor (Parameter)");
        final var info = new Info(fenster);
        farbmodellbeobachter.rundschreiben().addChangeListener(leinwand);
        verfolgung.addPropertyChangeListener("Zeiger", leinwand);
        leinwand.addPropertyChangeListener("Färbedauer", tafel);
        leinwand.addPropertyChangeListener("Restfärbedauer", tafel);
        leinwand.addPropertyChangeListener("Maldauer", tafel);
        leinwand.addPropertyChangeListener("Restmaldauer", tafel);
        kalkulator.rundschreiben().addChangeListener(leinwand);
        kalkulator.algorithmus().addPropertyChangeListener("Bildgröße", tafel);
        kalkulator.addPropertyChangeListener("Antialiasing", leinwand);
        kalkulator.addPropertyChangeListener("Vorschau", leinwand);
        kalkulator.addPropertyChangeListener("Auflösung", leinwand);
        kalkulator.addPropertyChangeListener("Bildgröße", leinwand);
        kalkulator.addPropertyChangeListener("Zeiger", leinwand);
        kalkulator.addPropertyChangeListener("Stopp", leinwand);
        kalkulator.addPropertyChangeListener("Berechnung", leinwand);
        kalkulator.addPropertyChangeListener("Zentrum", leinwand);
        kalkulator.addPropertyChangeListener("Auflösung", tafel);
        kalkulator.addPropertyChangeListener("Berechnung", tafel);
        kalkulator.addPropertyChangeListener("Iterationslimit", tafel);
        kalkulator.addPropertyChangeListener("Iterationsmaximum", tafel);
        kalkulator.addPropertyChangeListener("Antialiasing", tafel);
        kalkulator.addPropertyChangeListener("Bevorzugung", tafel);
        kalkulator.addPropertyChangeListener("Qualitaet", tafel);
        kalkulator.addPropertyChangeListener("Sichtfenster", tafel);
        kalkulator.addPropertyChangeListener("Bildgröße", tafel);
        kalkulator.addPropertyChangeListener("Rechendauer", tafel);
        kalkulator.addPropertyChangeListener("Iterationslimit", eingabe);
        kalkulator.addPropertyChangeListener("Auflösung", eingabe);
        kalkulator.addPropertyChangeListener("Sichtfenster", eingabe);
        kalkulator.addPropertyChangeListener("Stopp", fenster);
        kalkulator.addPropertyChangeListener("Berechnung", fenster);
        kalkulator.addPropertyChangeListener("Berechnung", verfolgung);
        kalkulator.addPropertyChangeListener("Berechnung", gestaltung);
        fenster.setLayout(null);
        fenster.setJMenuBar(menue(betrieb, fenster, kalkulator, definition, farbmodellbeobachter, leinwand, eingabe,
                ausfuhr, einfuhr, foto, gestaltung, tafel, pipette, zentrum, info, ende, verfolgung));
        fenster.setContentPane(leinwand);
        fenster.pack();
        if (betrieb == Betrieb.MINIMAL) {
            fenster.setMaximumSize(fenster.getPreferredSize());
        }
        fenster.setLocationByPlatform(true);
        fenster.setVisible(true);
        eingabe.setLocationRelativeTo(fenster);
        gestaltung.setLocationRelativeTo(fenster);
        tafel.setLocationRelativeTo(fenster);
        pipette.setLocationRelativeTo(fenster);
        zentrum.setLocationRelativeTo(fenster);
        info.setLocationRelativeTo(fenster);
    }

}
